; Copyright (c) 2010, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;	PRO p_write_csv, arg1, arg2, SEPERATOR=SEPERATOR
;
; PURPOSE:
;	Write csv data structure to a file
;
; CATEGORY:
;	Data import
;
; CALLING SEQUENCE:
; 	p_write_csv, arg1, arg2, SEPERATOR=SEPERATOR
;
; INPUTS:
;	arg1:		can be either a text string with a filename or a csvdata structure
;   arg2:		must be a csvdata structure if arg1 is a filename else undefined
;
; KEYWORD PARAMETERS:
;	SEPERATOR		Specify a seperator character (defaults to ',')
;
;
; EXAMPLE:
;	data = p_read_csv(pickfile())	; read data from file with output to data and the column names to names
;	p_write_csv, 'c:\temp\test.csv', data	; write it back to disk
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid   aug 2010
;-

PRO p_write_csv, arg1, arg2, SEPERATOR=SEPERATOR

	if n_elements(SEPERATOR) eq 0 then SEPERATOR = ','

	if n_elements(arg1) eq 0 then begin
		print, 'ERROR: You must specify either a csv data structure or a filename, csv data structure as input'
		return
	endif

	if size(arg1, /type) eq 7 then begin
		filename = arg1

		if n_elements(arg2) eq 0 then begin
			print, 'ERROR: You must specify a csv data structure as second argument'
			return
		endif

		if size(arg2, /type) ne 8 then begin
			print, 'ERROR: You must specify a csv data structure as second argument'
			return
		endif

		csv = arg2

		if csv.type ne 'csvdata' then begin
			print, 'WARNING: csv input data structure is not of type "csvdata"'
		endif
	endif else if size(arg1, /type) eq 8 then begin

		csv = arg1
		filename = csv.filename

		if csv.type ne 'csvdata' then begin
			print, 'WARNING: csv input data structure is not of type "csvdata"'
		endif

	endif else begin
		print, 'ERROR: illegal input to p_write_csv'
		return
	endelse

	; open file for writing
	openw, lun, filename, /GET_LUN

	; check if name row should be written
	do_write_names = 0
	for i=0, n_elements(csv.columns)-1 do begin
		if csv.columns[i] ne '' then begin
			if strmid(csv.columns[i], 0, 3) ne 'COL' then do_write_names = 1
		endif
	endfor

	; if column names exists then write these
	if do_write_names eq 1 then begin
		out = ''
		for i=0, n_elements(csv.columns)-1 do begin
			if i ne 0 then out += SEPERATOR
			out += csv.columns[i]
		endfor
		printf, lun, out
	endif

	; now write description row if defined
	if do_write_names ne 0 then begin
		do_write_desc = 0
		for i=0, n_elements(csv.descriptions)-1 do begin
			if csv.descriptions[i] ne '' then do_write_desc = 1
		endfor

		; if column names exists then write these
		if do_write_desc eq 1 then begin
			out = ''
			for i=0, n_elements(csv.descriptions)-1 do begin
				if i ne 0 then out += SEPERATOR
				out += csv.descriptions[i]
			endfor
			printf, lun, out
		endif
	endif

	; finally write the data values
	s = size(csv.data, /dim)
	for row = 0, s[1]-1 do begin
		out = ''
		for col = 0, s[0]-1 do begin
			if col ne 0 then out += SEPERATOR
			if csv.datatypes[col] eq 0 OR csv.datatypes[col] eq 7 then begin
				out += '"'
				out += csv.data[col, row]
				out += '"'
			endif else out += csv.data[col, row]
		endfor
		printf, lun, out
	endfor


	free_lun, lun


END