; Copyright (c) 2010, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;	FUNCTION p_csv_column, csv, column
;
; PURPOSE:
;	Extract column data from a csv data structure
;
; CATEGORY:
;	Data import
;
; CALLING SEQUENCE:
; 	Result = p_csv_column(csv, column)
;
; INPUTS:
;	csv:		is a data structure as returned by p_read_csv()
;	column		specify the column that is requested
;					can be either an integer value (interpreted as a column index)
;					os a string which is interpreted as a column name
;
; KEYWORD PARAMETERS:
;	DESCRIPTION:		Returns the description value of the column
; 	DATATYPE:			Returns the data type code of the column
;
; OUTPUTS:
;	Returns a vector containing the requested data on success
;           -1 on error
;
;
; EXAMPLE:
;	data = p_read_csv(pickfile())	; read data from file with output to data and the column names to names
;	plot, p_csv_column(data, 0)	; plot column 0
;	plot, p_csv_column(data, 'time')	; plot column name 'time'
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid   Aug 2010
;-

FUNCTION p_csv_column, csv, column, DESCRIPTION=DESCRIPTION, DATATYPE=DATATYPE

	if size(csv, /type) ne 8 then begin
		print, 'ILLEGAL input to p_csv_column.  Must be a csv structure as returned by p_read_csv'
		return, -1
	endif

	if csv.type ne 'csvdata' then print, 'WARNING input to p_csv_column is not of type csvdata (will probably fail)'

	; check for names column input
	if size(column, /type) eq 7 then begin
		s = where(csv.columns eq column, count)
		if count eq 0 then begin
			print, 'ERROR: Column ' + column + ' was not found in csv data'
			return, -1
		endif
		if n_elements(s) gt 1 then print, 'WARNING: Multiple columns of name ' + column + ' found - return the first!'

		column = s[0]
	endif


	DESCRIPTION = csv.descriptions[column]
	DATATYPE = csv.datatypes[column]


	return, fix(reform(csv.data[column, *]), TYPE=csv.datatypes[column])
END