; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;	Unauthorized reproduction prohibited.
;+
; NAME:
;	FUNCTION p_find_group_range, header, group, start_index, end_index, FULL_RANGE=FULL_RANGE, GROUP_ONLY=GROUP_ONLY, OBJECT_ONLY=OBJECT_ONLY
;
; PURPOSE:
;	Find index range within a PDS header of a specified GROUP or OBJECT
;  (Low level utility routine - not for normal use)
;
; CALLING SEQUENCE:
;	 Result = p_find_group_range(header, group, start_index, end_index)
;
; INPUTS:
;	header:			Valid PDS header structure
;	group:			Name of the group to search for either as command seperated tag list or as a string array
;   start_index:    Start index of the search
; 				    if start_index is undefined the default value 0 will be used
;   end_index:    	End index of the search
; 				    if end_index is undefined the last element in the header will be used as default
;
; OUTPUT:
;   start_index:    After a successfull search this parameter will contain the index of the begin group tag

;   start_index:    After a successfull search this parameter will contain the index of the end group tag
;
; RETURNS:
;		1 on success else 0
;
; KEYWORD PARAMETERS:
; 	FULL_RANGE:				If set the full range of the PDS header will be searched
;	GROUP_ONLY:				For the search to only look for GROUP tags
; 	OBJECT_ONLY:			For the search to only look for OBJECT tags
;
; EXAMPLE:
; 	p_read, pickfile(), h, d
; 	p_find_group_range, h, 'IMAGE', istart, iend
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid, 11/08-2004
;-

FUNCTION p_find_group_range, header, group, start_index, end_index, GROUP_ONLY=GROUP_ONLY, OBJECT_ONLY=OBJECT_ONLY

	; is valid header ?
	if p_is_header(header) eq 0 then return, 0

	; seperate path elements
	if n_elements(group) gt 1 then $
		nodes = group $
	else $
		nodes = strsplit(group, '.', /extract)

	; generate search indeces

	if keyword_set(GROUP_ONLY) then begin
		idx = where(header.tags eq 'G>' OR header.tags eq 'G<')
	endif else if keyword_set(OBJECT_ONLY) then begin
		idx = where(header.tags eq 'O>' OR header.tags eq 'O<')
	endif else begin
		idx = where(header.tags eq 'G>' OR header.tags eq 'G<' OR $
	            	header.tags eq 'O>' OR header.tags eq 'O<')
	endelse

	idx = idx[sort(idx)]


	level=0
	curr_level=0
	if idx[0] ge 0 then begin
		for i=0, n_elements(idx)-1 do begin
			tt = strmid(header.tags[idx[i]], 1, 1)
			if tt eq '>' then level = level + 1
			if tt eq '<' then level = level - 1

			if header.values[idx[i]] eq nodes[curr_level] then begin
				curr_level = curr_level + 1
				if curr_level eq n_elements(nodes) then begin
					start_index = idx[i]
					tlevel = level
					for j=i+1, n_elements(idx)-1 do begin
						tt = strmid(header.tags[idx[j]], 1, 1)
						if tt eq '>' then level = level + 1
						if tt eq '<' then level = level - 1
						if tlevel-1 eq level AND header.values[idx[j]] eq nodes[curr_level-1] then begin
							end_index = idx[j]
							return, 1
						endif
					endfor
				endif
			endif

			if level lt curr_level then return, 0
		endfor
	endif

	return, 0
END