; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;	Unauthorized reproduction prohibited.
;+
; NAME:
;	FUNCTION p_get_struct, header, tag, unit
;
; PURPOSE:
;	Read a PDS GROUP/OBJECT structure into an IDS structure variable
;   (Note internal utility routine. Should not be used by normal users.
;    Please use p_value instead)
;
; CALLING SEQUENCE:
;	Result = p_get_struct(header, tag, unit)
;
; INPUTS:
;	header:	Valid PDS header structure
;   tag:    the path to the GROUP/OBJECT structure within the PDS header
;
; OUTPUT
;   unit:	At the moment nothing, but in future version this variable will contain the unit data
;		    from the header is some form
;
; RETURNS:
;	IDL structure containing the header data
;
; KEYWORD PARAMETERS:
;	INDEX_FIRST    : return the header index of the start of the structure (the G> tag)
; INDEX_LAST     : return the header index of the end of the structure (the G< tag)
; AS_STRING      : Return all values as string type
;
; OUTPUTS:
;	NONE
;
; EXAMPLE:
;   p_read, pickfile(), h, d
;   str = p_get_struct(h, 'IMAGE')
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid, 11/8-2004
;-

FUNCTION p_get_struct, header, tag, unit, INDEX_FIRST=INDEX_FIRST, INDEX_LAST=INDEX_LAST, AS_STRING=AS_STRING

	nodes = strsplit(tag, '.', /extract)
	str_name = nodes[n_elements(nodes)-1]
	istart = 0
	iend = n_elements(header.tags)-1

	if p_find_group_range(header, tag, istart, iend) eq 0 then begin
	  print, 'group/object not found in the label: ' + strtrim(tag, 2)
	  return, 0
  endif
	
	; constrain to just the content of the structure	
	istart = istart + 1
	iend   = iend - 1
	if iend ge n_elements(header.tags) then iend = n_elements(header.tags)-1

  if iend lt istart then begin
    print, 'Illegal range found!!'
    return, 0
  endif
  
  ; return the search index range
  INDEX_FIRST = istart - 1
  INDEX_LAST  = iend + 1

	for i=istart, iend do begin
		if header.tags[i] eq 'G>' OR header.tags[i] eq 'O>' then begin
		  ; extract the tage name
			sname = header.values[i]
			; extract the sub-structure value
			value = p_get_struct(header, tag + '.' + sname, INDEX_FIRST=tstart, INDEX_LAST=tend)
			
			; shift the loop index to the last entry in the group
			i = tend 
			
			; add the sub structure to the output structure
			ret = p_set_struct_tag(ret, sname, value)
			
		endif else if header.tags[i] eq 'G<' OR header.tags[i] eq 'O<' then begin
			; do nothing with new implementation (end group/object)
		endif else begin
			sname = p_remove_tag_namespace(header.tags[i], /IDL_COMPATIBLE_POINTER)
			if sname ne '' then begin		  
		    svalue = header.values[i]
        ; normal value tag
        if keyword_set(AS_STRING) then begin
          value = svalue
        endif else begin      
          svalue = p_pds_to_idl_var_string(svalue, unit)
          if n_elements(svalue) gt 1 then begin
            type = p_type_of_string(svalue[0])
            value = fix(svalue, TYPE=type)
          endif else begin
            type = p_type_of_string(svalue)
            value = fix(svalue, TYPE=type)
          endelse
        endelse
		
		    ; add the sub structure to the output structure
        ret = p_set_struct_tag(ret, sname, value)
      endif
    endelse
	endfor

  ; if we actually extracted something than return this
  if n_elements(ret) ne 0 then return, ret
  
  ; else
  print, 'No label data found for ' + strtrim(tag, 2)
  return, 0  
END