; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;   FUNCTION p_value, header, tag, unit
;
; PURPOSE:
;   Extract a header value from a PDS header
;
; CALLING SEQUENCE:
;   Result = p_value(header, tag, unit)
;
; INPUTS:
;   header:   Valid PDS header structure
;   tag:    text string with the header value requested   (Objects and groups are referenced
;                                                          using the syntax <group name>.tag
;                                                          Exmaple:  'IMAGE.MEAN'
;                                      If tag refers to a GROUP or OBJECT struct
;                                      The value is returned as an IDL structure
;
; KEYWORD PARAMETERS:
;   FOUND:      output 1 if the tag was found else 0
;   INDEX:      output index of tag if found
;   AS_STRING:  if set the value is always returnes as a text string
;   FIND:       if set the label is searched for the first label named <tag>
;   HISTORY:	If set look for value in history label not main label
;
; OUTPUTS:
;   unit: Will containg the PDS defined unit of the header item
;
; RETURNS:
;   value of header entry   - '' on error
;
; EXAMPLE:
;   p_read, pickfile(), h, d  ; read PDS file
;   print, p_value(h, 'IMAGE.MEAN', unit)   ; read single tag
;   print, p_value(h, 'IMAGE')          ; read full pbject to structure
;
; MODIFICATION HISTORY:
;   Written by:  Stubbe F. Hviid, 12/05-2004
;-

FUNCTION pds_value_str, header, tag_path, FOUND=FOUND, IS_GROUP=IS_GROUP, ENFORCE_NAMESPACE=ENFORCE_NAMESPACE, INDEX=INDEX

  FOUND = 0
  IS_GROUP = 0

  ; verify that header is a valid PDS header
  if p_is_header(header) eq 0 then return, ''

  ; parse the tag_path
  p_parse_tag_path, tag_path, nodes, name, ROOT_LEVEL=is_root

  ; check if the tag path specifies a namespace
  if strpos(name, ':') gt 0 then ENFORCE_NAMESPACE = 1

  ; find search range
  start_index = 0
  end_index = n_elements(header.tags)-1

  ;first check if the tag refders to a group
  if p_find_group_range(header, tag_path, start_index, end_index) then begin
    FOUND = 1
    IS_GROUP = 1
    INDEX = start_index
    return, tag_path
  endif

  ; limit search range if group statement
  if is_root eq 0 then begin
    if p_find_group_range(header, nodes, start_index, end_index) eq 0 then begin
      FOUND = 0
      return, ''
    endif
  endif

  ; do range check
  if strmid(header.tags[start_index], 1, 1) eq '>' then start_index = start_index + 1
  if strmid(header.tags[end_index], 1, 1) eq '<' then end_index = end_index - 1

  ; look for header item
  level = 0
  for i=start_index, end_index do begin
    isg = strmid(header.tags[i], 1, 1)
    if isg eq '>' then level = level + 1
    if isg eq '<' then level = level - 1
    if level eq 0 then begin
      tst_value = header.tags[i]
      if isg eq '>' then tst_value = header.values[i]

      if tst_value eq name then begin
        FOUND = 1
        IS_GROUP = 1
        INDEX = i
        return, header.values[i]
        endif

      if NOT keyword_set(ENFORCE_NAMESPACE) then begin
        nntag = p_remove_tag_namespace(header.tags[i])
        if nntag eq name then begin
          FOUND=1
          INDEX=i
          return, header.values[i]
        endif
      endif
    endif
  endfor

  FOUND = 0
  return, ''
END

FUNCTION p_value, header, tag, unit, FOUND=FOUND, INDEX=INDEX, _EXTRA=_EXTRA, AS_STRING=AS_STRING, FIND=FIND, HISTORY=HISTORY

  if keyword_set(HISTORY) then begin
  	if p_struct_tag_exists(header, 'history') eq 0 then begin
  		FOUND = 0
  		return, ''
  	endif
  	ttag = tag
  	if strupcase(strmid(tag, 0, 8)) ne 'HISTORY.' then ttag = 'HISTORY.' + tag
	return, p_value(header.history, ttag, unit, FOUND=FOUND, INDEX=INDEX, AS_STRING=AS_STRING, FIND=FIND)
  endif

  ; the normal handling
  tag = p_uax2pds(tag)

  ; first handle search mode
  if keyword_set(FIND) then begin
    s = where(header.tags eq tag)

    if s[0] ge 0 then begin
      if keyword_set(AS_STRING) then return, value
      svalue = p_pds_to_idl_var_string(header.values[s[0]], unit)
      if n_elements(svalue) gt 1 then begin
        type = p_type_of_string(svalue[0])
        return, fix(svalue, TYPE=type)
      endif else begin
        type = p_type_of_string(svalue)
        return, fix(svalue, TYPE=type)
      endelse
    endif
    print, 'Tag: ' + tag + ' not found in label'
    return, ''
  endif

  value = pds_value_str(header, tag, FOUND=FOUND, INDEX=INDEX, _EXTRA=_EXTRA)

    if value eq '' then return, ''

    value = strtrim(value, 2)


    ; handle structures seperately
    if p_is_group(header, tag) ne 0 then begin
       return, p_get_struct(header, tag, unit, _EXTRA=_EXTRA)
    endif

	if keyword_set(AS_STRING) then return, value

    svalue = p_pds_to_idl_var_string(value, unit)

	if n_elements(svalue) gt 1 then begin
		type = p_type_of_string(svalue[0])
		ret = fix(svalue, TYPE=type)

	endif else begin
    	type = p_type_of_string(svalue)
		ret = fix(svalue, TYPE=type)
    endelse


    return, ret
END